defmodule Console.Ecran.Parties do
  @moduledoc """
  Écran appelé pour afficher les parties existantes.
  """

  @behaviour Console.Ecran

  defstruct nom: :inconnu, choix: []

  @typedoc """
  Un écran `Console.Ecran.Parties`.
  """
  @type t() :: %{nom: String.t(), choix: %{String.t() => {integer(), pid()}}}

  @doc """
  Crée un écran.
  """
  @spec new(String.t()) :: t()
  def new(nom) do
    choix =
      Enum.map(Jeu.lister_parties(), fn {id, {processus, titre}} -> {id, processus, titre} end)

    %Console.Ecran.Parties{nom: nom, choix: choix}
  end

  @impl true
  @doc """
  Retourne le titre de l'écran.
  """
  @spec titre(struct()) :: String.t()
  def titre(_écran), do: "Parties de jeux de cartes actuelles"

  @impl true
  @doc """
  Retourne le texte de l'écran.
  """
  @spec texte(struct()) :: String.t()
  def texte(écran) do
    titres =
      for {{_, _, titre}, index} <- Enum.with_index(écran.choix) do
        "#{index + 1} - #{titre}"
      end

    titres = Enum.join(titres, "\n")

    """
    Bienvenue, #{écran.nom} !

    Pour jouer à une partie déjà créée, entrez simplement son numéro.
    Sinon, entrez C pour créer une nouvelle partie.

    Parties actuellement enregistrées :

    #{titres}
    """
  end

  @impl true
  @doc """
  Retourne le prompt de l'écran (le texte tout en bas, indiquant quelle information est à préciser ici).
  """
  @spec prompt(struct()) :: String.t()
  def prompt(_écran), do: "Entrez le numéro de la partie à rejoindre ou C pour en créer une :"

  @impl true
  @doc """
  Gère les entrées claviers de l'utilisateur.

  Cette fonction peut retourner plusieurs informations :

  - `:silence` : indique à l'interface qu'il n'est pas nécessaire d'afficher l'écran de novueau ;
  - `:prompt` : indique à l'interface qu'il est simplement nécessaire d'afficher de nouveau le prompt de l'écran, sans son texte ;
  - `:rafraîchir` : indique à l'interface qu'il faut afficher tout l'écran (titre, texte et prompt) ;
  - `{atome, texte}` : où `atome` est l'un des trois atomes ci-dessus et `texte` est le texte à afficher ;
  - `écran` : où 'écran' est la nouvelle structure du module `écran`.

  """
  @spec gérer_entrée(struct(), String.t()) :: Console.Ecran.retour_clavier()
  def gérer_entrée(écran, "C") do
    règles = Jeu.Huit
    {id, processus} = Jeu.créer_partie(règles)
    {:ok, joueur} = Jeu.ajouter_joueur(id, processus, écran.nom)

    Console.Ecran.Plateau.new(
      %{nom: écran.nom, id: joueur},
      %{id: id, processus: processus, titre: règles.titre()}
    )
  end

  def gérer_entrée(écran, entrée) do
    with {index, _} <- Integer.parse(entrée),
         {:ok, choix} <- Enum.fetch(écran.choix, index - 1) do
      {id, processus, titre} = choix

      case Jeu.ajouter_joueur(id, processus, écran.nom) do
        {:ok, joueur} ->
          Console.Ecran.Plateau.new(
            %{nom: écran.nom, id: joueur},
            %{id: id, processus: processus, titre: titre}
          )

        :invalide ->
          {:prompt, "Le joueur n'a pas pu être ajouté dans cette partie."}
      end
    else
      :error -> {:prompt, "Cette partie n'est pas valide."}
    end
  end
end
